/*
  RS485 Modbus Temperature and Humidity Sensor Reader
  Made by KinCony: https://www.kincony.com
  
  This program reads temperature and humidity data from a Modbus RS485 sensor
  and prints the values via the Serial monitor.

  Hardware configuration:
  - RS485 communication using GPIO TX (32) and RX (35)
  - Modbus RTU protocol
  - Temperature and humidity values are stored in separate registers

  You must install the ModbusMaster library in Arduino IDE.
*/

#include <ModbusMaster.h>  // Include the ModbusMaster library for Modbus communication

// Define GPIO pins for RS485 communication (TX and RX)
#define TXD_PIN 32  // Transmit pin for RS485
#define RXD_PIN 35  // Receive pin for RS485

// Define the Modbus slave ID (this should match the sensor's address)
#define SLAVE_ID 1

// Create a ModbusMaster object to handle Modbus communication
ModbusMaster node;

/*
  This function is called before starting a Modbus transmission.
  If your hardware needs control of RS485 DE/RE pins, configure them here.
*/
void preTransmission()
{
  // Optional: Control DE/RE pins for RS485 if necessary
}

/*
  This function is called after completing a Modbus transmission.
  If your hardware needs to reset DE/RE pins, configure them here.
*/
void postTransmission()
{
  // Optional: Control DE/RE pins for RS485 if necessary
}

void setup()
{
  // Start the Serial communication for debugging at 9600 baud rate
  Serial.begin(9600);
  
  // Start Serial2 for RS485 communication using GPIO 32 (TX) and GPIO 35 (RX) at 9600 baud rate
  Serial2.begin(9600, SERIAL_8N1, RXD_PIN, TXD_PIN);

  // Initialize the Modbus communication with the slave ID and associate it with Serial2
  node.begin(SLAVE_ID, Serial2);

  // Set the pre- and post-transmission handlers for controlling RS485 hardware if needed
  node.preTransmission(preTransmission);
  node.postTransmission(postTransmission);

  // Print an initial message to Serial monitor
  Serial.println("RS485 Modbus Sensor Example - Reading Temperature and Humidity");
}

void loop()
{
  uint8_t result;  // Variable to store the result of Modbus operations
  uint16_t data[2];  // Array to store the raw temperature and humidity data from the sensor

  // Read two input registers starting from address 0x0000, where temperature and humidity are stored
  result = node.readInputRegisters(0x0000, 2);  // Read 2 registers (temperature and humidity)

  // Check if the read operation was successful
  if (result == node.ku8MBSuccess)
  {
    // Retrieve raw temperature and humidity data from Modbus response buffer
    uint16_t temperatureRaw = node.getResponseBuffer(0);  // Temperature is in the first register
    uint16_t humidityRaw = node.getResponseBuffer(1);     // Humidity is in the second register

    // Parse temperature value
    float temperature;
    if (temperatureRaw < 10000)  // Check if the value represents a positive temperature
      temperature = temperatureRaw * 0.1;  // Positive temperature
    else
      temperature = -1 * (temperatureRaw - 10000) * 0.1;  // Negative temperature

    // Parse humidity value (always positive)
    float humidity = humidityRaw * 0.1;

    // Print the parsed temperature value to the Serial monitor
    Serial.print("Temperature: ");
    Serial.print(temperature);
    Serial.println(" °C");

    // Print the parsed humidity value to the Serial monitor
    Serial.print("Humidity: ");
    Serial.print(humidity);
    Serial.println(" %RH");
  }
  else
  {
    // If the read operation failed, print the error code to the Serial monitor
    Serial.print("Failed to read from sensor, error: ");
    Serial.println(result);
  }

  // Wait for 5 seconds before reading the sensor again
  delay(5000);
}
