/*
  Made by KinCony IoT: https://www.kincony.com

  Program functionality:
  This program uses an ESP32-S3 to read inputs from two XL9535 I/O expander chips 
  (with I2C addresses 0x24 and 0x25) and one PCF8574 I/O expander (with I2C address 0x23) via I2C. 
  The XL9535 chips handle 1-32 channels, while the PCF8574 chip handles channels 33-40. 
  The program reads the state of all these inputs and prints them in binary format to the serial monitor.

  Key points:
  - The I2C bus is initialized on GPIO pins 11 (SDA) and 10 (SCL) with a frequency of 40kHz.
  - The XL9535 chips read the state of input pins for channels 1-32.
  - The PCF8574 chip reads the state of input pins for channels 33-40.
  - The state of the inputs is printed to the serial monitor in binary format every second.
*/

#include <PCA95x5.h>
#include <Wire.h>

// Initialize the PCA9555 objects for XL9535 chips (1-32 channels)
PCA9555 ioex1;  // For channels 1-16 (I2C address 0x24)
PCA9555 ioex2;  // For channels 17-32 (I2C address 0x25)

// Define the I2C address for the PCF8574 chip (for channels 33-40)
const int pcf8574_address = 0x23;

void setup() {
    // Start serial communication for debugging
    Serial.begin(115200);
    delay(2000);  // Wait for 2 seconds to ensure everything is ready

    // Initialize the I2C bus with GPIO 11 as SDA and GPIO 10 as SCL, 40kHz frequency
    Wire.begin(11, 10, 40000);

    // Attach the first PCA9555 device (for channels 1-16)
    ioex1.attach(Wire, 0x24);
    ioex1.polarity(PCA95x5::Polarity::ORIGINAL_ALL);
    ioex1.direction(PCA95x5::Direction::IN_ALL);  // Set all pins as inputs

    // Attach the second PCA9555 device (for channels 17-32)
    ioex2.attach(Wire, 0x25);
    ioex2.polarity(PCA95x5::Polarity::ORIGINAL_ALL);
    ioex2.direction(PCA95x5::Direction::IN_ALL);  // Set all pins as inputs
}

void loop() {
    // Read and print the state of inputs from the first PCA9555 (channels 1-16)
    Serial.print("1-16 input states: ");
    Serial.println(ioex1.read(), BIN);

    // Read and print the state of inputs from the second PCA9555 (channels 17-32)
    Serial.print("17-32 input states: ");
    Serial.println(ioex2.read(), BIN);

    // Read and print the state of inputs from the PCF8574 (channels 33-40)
    Serial.print("33-40 input states: ");
    byte pcf8574_state = readPCF8574();
    Serial.println(pcf8574_state, BIN);

    delay(1000);  // Wait for 1 second before reading again
}

// Function to read the state of the PCF8574 (channels 33-40)
byte readPCF8574() {
    Wire.requestFrom(pcf8574_address, 1);  // Request 1 byte from PCF8574

    if (Wire.available()) {
        return Wire.read();  // Return the byte representing the state of inputs
    } else {
        return 0xFF;  // Return 0xFF if no data is available (all inputs are high)
    }
}
