/*
  PCF8574 I2C Digital I/O Expander Example
  Made by KinCony IoT: https://www.kincony.com

  Description:
  This program demonstrates how to use a PCF8574 I2C expander to read input pins and control relays.
  It reads input states from a PCF8574 module and controls corresponding relays.
  
  The program initializes the input pins as INPUT and output pins as OUTPUT.
  When an input is triggered (LOW), the corresponding relay (output) will activate (LOW).
  It also prevents the relays from briefly activating on startup.

  Custom I2C pins are defined for SDA and SCL.

  License:
  You are free to use and modify this code for personal or commercial purposes as long as this
  copyright notice remains intact.
*/

// Define custom I2C pins for communication
#define SDA_PIN 12    // Set SDA pin for I2C communication
#define SCL_PIN 11   // Set SCL pin for I2C communication

#include "Arduino.h"
#include "PCF8574.h"
#include <Wire.h>

// Define I2C addresses for the input and output PCF8574 modules
PCF8574 pcf8574_input(0x22);  // I2C address for the input expander (change according to your setup)
PCF8574 pcf8574_output(0x24); // I2C address for the output expander (change according to your setup)

void setup() {
  // Start serial communication for debugging purposes
  Serial.begin(115200);
  delay(1000);  // Wait for serial communication to initialize

  // Initialize I2C and set custom SDA and SCL pins
  Wire.begin(SDA_PIN, SCL_PIN);

  // Ensure all relay outputs are set to HIGH (deactivated) before initializing the expander
  // This step is to avoid any relay from briefly turning on during setup
  for (int i = 0; i < 6; i++) {
    pcf8574_output.digitalWrite(i, HIGH);
  }

  // Initialize input pins as INPUT
  pcf8574_input.pinMode(P0, INPUT);
  pcf8574_input.pinMode(P1, INPUT);
  pcf8574_input.pinMode(P2, INPUT);
  pcf8574_input.pinMode(P3, INPUT);
  pcf8574_input.pinMode(P4, INPUT);
  pcf8574_input.pinMode(P5, INPUT);

  // Initialize output pins as OUTPUT (these control the relays)
  pcf8574_output.pinMode(P0, OUTPUT);
  pcf8574_output.pinMode(P1, OUTPUT);
  pcf8574_output.pinMode(P2, OUTPUT);
  pcf8574_output.pinMode(P3, OUTPUT);
  pcf8574_output.pinMode(P4, OUTPUT);
  pcf8574_output.pinMode(P5, OUTPUT);

  // Ensure all outputs (relays) are deactivated at the start (set to HIGH)
  for (int i = 0; i < 6; i++) {
    pcf8574_output.digitalWrite(i, HIGH);
  }

  // Initialize the PCF8574 modules and check for errors
  Serial.print("Init pcf8574_input...");
  if (pcf8574_input.begin()) {
    Serial.println("OK");  // Input expander initialized successfully
  } else {
    Serial.println("KO");  // Input expander initialization failed
  }

  Serial.print("Init pcf8574_output...");
  if (pcf8574_output.begin()) {
    Serial.println("OK");  // Output expander initialized successfully
  } else {
    Serial.println("KO");  // Output expander initialization failed
  }
}

void loop() {
  // Read input states from the input expander (PCF8574)
  uint8_t val[6];  // Array to store input states
  for (int i = 0; i < 6; i++) {
    val[i] = pcf8574_input.digitalRead(i);  // Read the state of each input pin
  }

  // Control relays based on the input states
  for (int i = 0; i < 6; i++) {
    if (val[i] == LOW) {
      // If the input is LOW (triggered), activate the corresponding relay (set to LOW)
      pcf8574_output.digitalWrite(i, LOW);
    } else {
      // If the input is HIGH (not triggered), deactivate the corresponding relay (set to HIGH)
      pcf8574_output.digitalWrite(i, HIGH);
    }
  }

  delay(300);  // Small delay to prevent rapid state changes
}
