/*
  Version: 1.0.0
  Created by KinCony (https://www.kincony.com)
  Description: 
    This program controls two PCF8574 I/O expanders connected via I2C to create a chase effect 
    across 16 output pins (P0-P7 on PCF8574_1 and PCF8574_2). 
    The effect consists of sequentially turning on and off LEDs to create a "running light" or "chase" effect.
  Date: 2024-12-24
*/

#include "Arduino.h"
#include "PCF8574.h"

// Set i2c addresses for the two PCF8574 devices
PCF8574 pcf8574_1(0x24);  // PCF8574 at address 0x24
PCF8574 pcf8574_2(0x25);  // PCF8574 at address 0x25

// Setup function, runs once when the program starts
void setup() {
  Wire.begin(9, 10); // SDA: GPIO9, SCL: GPIO10 (for I2C communication)
  Serial.begin(115200); // Start serial communication for debugging

  // Set pinMode to OUTPUT for all 16 pins on both PCF8574 chips
  for (int i = 0; i < 8; i++) {
    pcf8574_1.pinMode(i, OUTPUT);  // Set pins P0-P7 of pcf8574_1 to OUTPUT
    pcf8574_2.pinMode(i, OUTPUT);  // Set pins P0-P7 of pcf8574_2 to OUTPUT
  }

  // Initialize PCF8574_1 and check if it's successful
  Serial.print("Init pcf8574_1...");
  if (pcf8574_1.begin()) {
    Serial.println("PCF8574_1_OK");
  } else {
    Serial.println("PCF8574_1_KO");
  }

  // Initialize PCF8574_2 and check if it's successful
  Serial.print("Init pcf8574_2...");
  if (pcf8574_2.begin()) {
    Serial.println("PCF8574_2_OK");
  } else {
    Serial.println("PCF8574_2_KO");
  }
}

// Loop function, runs repeatedly
void loop() {
  // Create the "chase" effect by turning on one LED at a time

  // First pass: Turn on pins one by one from P0 to P7 on both PCF8574 chips
  for (int i = 0; i < 8; i++) {
    pcf8574_1.digitalWrite(i, HIGH);  // Turn on the LED on pcf8574_1 (pin P0 to P7)
    delay(100);                       // Delay for 100ms to create the "chase" effect
    pcf8574_1.digitalWrite(i, LOW);   // Turn off the LED on pcf8574_1

    // Similarly, turn on LEDs on pcf8574_2 from P0 to P7
    if (i < 8) {
      pcf8574_2.digitalWrite(i, HIGH);  // Turn on the LED on pcf8574_2
      delay(100);                       // Delay for 100ms
      pcf8574_2.digitalWrite(i, LOW);   // Turn off the LED on pcf8574_2
    }
  }

  // Optionally reverse the sequence for a full "chase" effect
  // Second pass: Turn on LEDs in reverse order from P7 to P0
  for (int i = 7; i >= 0; i--) {
    pcf8574_1.digitalWrite(i, HIGH);  // Turn on the LED on pcf8574_1 (pin P7 to P0)
    delay(100);                       // Delay for 100ms
    pcf8574_1.digitalWrite(i, LOW);   // Turn off the LED on pcf8574_1

    // Similarly, reverse the LEDs on pcf8574_2 from P7 to P0
    if (i < 8) {
      pcf8574_2.digitalWrite(i, HIGH);  // Turn on the LED on pcf8574_2
      delay(100);                       // Delay for 100ms
      pcf8574_2.digitalWrite(i, LOW);   // Turn off the LED on pcf8574_2
    }
  }
}
